<#
    .SYNOPSIS
        Creates a Windows 10 Store-ready APPX package from an application's command-line installer
    .DESCRIPTION
        This converter depends on an optional Windows feature being enabled. Please refer to machine setup instructions to ensure you have enabled it.

        NOTE: Your installer must be able to run unattended/silently. Sequencing an attended installer, or one with any UI is unsupported at this time.
    .PARAMETER Setup
        Use this flag to run DesktopAppConverter in setup mode. Setup mode supports expanding a provided base image.
    .PARAMETER BaseImage
        The path to an unexpanded base image. This param is needed if -Setup is specified.
    .PARAMETER ExpandedBaseImage
        The path to an already expanded base image. This param is used when -Setup is not specified (ie you already have an expanded image).
    .PARAMETER Installer
        The path to the installer for your application - must be able to run unattended/silently
    .PARAMETER InstallerArguments
        [optional] A comma-separated list or string of arguments to force your installer to run unattended/silently. This parameter is optional if your installer is an msi.
        In order to get a log from your installer, you must supply the logging argument for the installer here and use the path "<log_folder>", which is a token that the converter will replace with the appropriate path.

        NOTE: The unattended/silent flags and log arguments will vary between installer technologies

        An example usage for this parameter: -InstallerArguments "/silent /log <log_folder>\install.log"
        Another example that doesn't produce a log file may look like: -InstallerArguments "/quiet", "/norestart"

        Again, you must literally direct any logs to the token path "<log_folder>" if you want the converter to capture it and put it in the final log folder
    .PARAMETER InstallerValidExitCodes
        [optional] A comma-separated list of exit codes that indicate your installer ran successfully (for example: 0, 1234, 5678) . By default this is 0 for non-msi, and 0, 1641, 3010 for msi.
    .PARAMETER Destination
        The desired destination for the converter's appx output - DesktopAppConverter can create this location if it doesn't already exist
    .PARAMETER PackageName
        The name of your Universal Windows App package
    .PARAMETER Publisher
        The publisher of your Universal Windows App package
    .PARAMETER Version
        The version number for your Universal Windows App package
    .PARAMETER AppExecutable
        [optional] The full path to your application's main executable if it were installed (it doesn't have to be) (eg "C:\Program Files (x86)\MyApp\MyApp.exe")
    .PARAMETER AppFileTypes
        [optional] A comma-separated list of file types which the application will be associated with (eg. ".txt, .doc", without the quotes).
    .PARAMETER AppId
        [optional It specifies a value to set Application Id to in the appx manifest. If it is not specified, it will be set to the value passed in for PackageName
    .PARAMETER AppDisplayName
        [optional It specifies a value to set Application Display Name to in the appx manifest. If it is not specified, it will be set to the value passed in for PackageName
    .PARAMETER AppDescription
        [optional It specifies a value to set Application Description to in the appx manifest. If it is not specified, it will be set to the value passed in for PackageName
    .PARAMETER PackageDisplayName
        [optional It specifies a value to set Package Display Name to in the appx manifest. If it is not specified, it will be set to the value passed in for PackageName
    .PARAMETER PackagePublisherDisplayName
        [optional It specifies a value to set Package Publisher Display Name to in the appx manifest. If it is not specified, it will be set to the value passed in for Publisher
    .PARAMETER MakeAppx
        [optional A switch that when present, tells this script to call MakeAppx on the output
    .PARAMETER NatSubnetPrefix
        [optional Prefix value to be used for the Nat instance. Typically you would want to change this only if your host
        machine is attached to the same subnet range as the Converter's NetNat. You can query the current Converter NetNat config using
        Get-NetNat cmdlet.
    .PARAMETER LogFile
        [optional] Specifies a log file. If omitted, a log file temporary location will be created.
    .EXAMPLE
        .\DesktopAppConverter.ps1 -ExpandedBaseImage C:\Images\Client -Installer C:\MyInstallers\MyApp.msi -Destination C:\MyAppxFolder -PackageName "MyApp" -Publisher "CN=Microsoft" -Version 1.0.0.0 -AppExecutable "C:\Program Files\MyApp\MyApp.exe" -MakeAppx

        Sequence an msi, put the specified AppExecutable into the AppxManifest.xml for your app, and package everything into "MyApp.appx".

    .EXAMPLE
        .\DesktopAppConverter.ps1 -ExpandedBaseImage C:\Images\Client -Installer C:\MyInstallers\MyAppSetup.exe -InstallerArguments "/quiet","/norestart" -Destination C:\MyAppxFolder -PackageName "MyApp" -Publisher "CN=Microsoft" -Version 1.0.0.0 -Verbose

        Sequence a setup exe installer by providing silent flags (NOTE: Flags will vary based on your installer technology),
        and leave the output unpackaged (you can manually package using makeappx.exe). The "-Verbose" flag causes progress to be printed to console.
#>
[CmdletBinding(DefaultParameterSetName="Convert")]
Param(
    [Parameter(Mandatory=$True, ParameterSetName="Setup")]
    [switch]
    $Setup,

    [Parameter(Mandatory=$True, ParameterSetName="Setup")]
    [string]
    [ValidateNotNullOrEmpty()]
    $BaseImage,

    [Parameter(Mandatory=$True, ParameterSetName="Convert")]
    [string]
    [ValidateNotNullOrEmpty()]
    $ExpandedBaseImage,

    [Parameter(Mandatory=$True, ParameterSetName="Convert")]
    [string]
    [ValidateNotNullOrEmpty()]
    $Installer,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [string[]]
    $InstallerArguments,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [int[]]
    $InstallerValidExitCodes,

    [Parameter(Mandatory=$True, ParameterSetName="Convert")]
    [string]
    [ValidateNotNullOrEmpty()]
    $Destination,

    [Parameter(Mandatory=$True, ParameterSetName="Convert")]
    [string]
    [ValidateNotNullOrEmpty()]
    $PackageName,

    [Parameter(Mandatory=$True, ParameterSetName="Convert")]
    [string]
    [ValidateNotNullOrEmpty()]
    $Publisher,

    [Parameter(Mandatory=$True, ParameterSetName="Convert")]
    [version]
    [ValidateNotNull()]
    $Version,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [string]
    $AppExecutable,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [string[]]
    $AppFileTypes,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [string]
    $AppId,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [string]
    $AppDisplayName,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [string]
    $AppDescription,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [string]
    $PackageDisplayName,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [string]
    $PackagePublisherDisplayName,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [switch]
    $MakeAppx,

    [Parameter(Mandatory=$False, ParameterSetName="Convert")]
    [string]
    $NatSubnetPrefix,

    [Parameter(Mandatory=$False)]
    [string]
    $LogFile
)

function CreateTempLocation
{
    $dacPath = Join-Path $env:SystemDrive "DesktopAppConverter"
    if (!(Test-Path $dacPath))
    {
        New-Item -Path $dacPath -ItemType Directory -Force > $null
    }
    $tempDirName = [System.Guid]::NewGuid().toString()
    $tempDirPath = Join-Path $dacPath $tempDirName
    $tempDir = New-Item -Path $tempDirPath -ItemType Directory -Force
    $tempDir.FullName
}

$ErrorActionPreference = "Stop"

$myLocation = (Split-Path (Get-Variable MyInvocation -Scope 0).Value.mycommand.path)

. (Join-Path $myLocation "converter_util\CWACLogger.ps1")

$hostTempLocation = CreateTempLocation
$logDirectory = $null

if (!$LogFile)
{
    $logDirectory = (New-Item -Path (Join-Path $hostTempLocation "logs") -ItemType Directory -Force).FullName
    $LogFile = Join-Path $logDirectory "DesktopAppConverter.log"
}
else
{
    New-Item -Path $LogFile -ItemType File -Force > $null
    $logDirectory = (Get-Item $LogFile).DirectoryName
}

$logger = [CWACLogger]::new($LogFile, (Join-Path $myLocation "telemetry"))
$logger.LogDiag("Log files can be found in $logDirectory")

# Log version if this was an official build
$versionFile = (Join-path $myLocation "version.txt")
if (Test-Path $versionFile)
{
    $versionString = Get-Content $versionFile
    $logger.LogVersion($versionString)
}

try
{
    . (Join-Path $myLocation "converter_util\EnvironmentAssertions.ps1")
    $logger.NewLogSection("Checking Prerequisites", "CheckPrereq")

    Assert-MinimumWindowsVersion $logger
    Assert-RequiredFeatureEnabled $logger

    $hcsDllPath = Join-Path $myLocation "isolation_manager"

    if ($Setup)
    {
        $logger.NewLogSection("Expanding Base Image", "ExpandBaseImage")
        $logger.SendFileParameter("BaseImage", $BaseImage)
        $logger.SendFileParameter("LogFile", $LogFile)

        $BaseImage = $BaseImage.Replace("`"","")
        . (Join-Path $myLocation "converter_util\Setup.ps1")
        $ExpandedBaseImage = Expand-BaseImage -BaseImage $BaseImage -DllPath $hcsDllPath -Logger $logger | Select-Object -Last 1
        $logger.LogDiag("Base image $BaseImage was expanded to $ExpandedBaseImage")
        return
    }

    $logger.SendParameter("AppId", $AppId)
    $logger.SendParameter("AppDisplayName", $AppDisplayName)
    $logger.SendParameter("AppDescription", $AppDescription)
    $logger.SendFileParameter("ExpandedBaseImage", $ExpandedBaseImage)
    $logger.SendFileSize("Installer", $Installer)
    $logger.SendFileParameter("Installer", $Installer)
    $logger.SendParameter("InstallerArguments", $InstallerArguments)
    $logger.SendParameter("InstallerValidExitCodes", $InstallerValidExitCodes)
    # There's no value in sending the Destination Command Line parameter, since it's required and is just a filepath.
    $logger.SendParameter("PackageDisplayName", $PackageDisplayName)
    $logger.SendParameter("PackageName", $PackageName)
    $logger.SendParameter("PackagePublisherDisplayName", $PackagePublisherDisplayName)
    $logger.SendParameter("Publisher", $Publisher)
    $logger.SendParameter("Version", $Version)
    $logger.SendFileParameter("AppExecutable", $AppExecutable)
    $logger.SendParameter("AppFileTypes", $AppFileTypes)
    $logger.SendSwitchParameter("MakeAppx", $MakeAppx)
    $logger.SendParameter("NatSubnetPrefix", $NatSubnetPrefix)
    $logger.SendFileParameter("LogFile", $LogFile)

    . (Join-Path $myLocation "converter_util\InputValidation.ps1")
    . (Join-Path $myLocation "converter_util\ManifestOps.ps1")
    . (Join-Path $myLocation "converter_util\Sequencer.ps1")
    . (Join-Path $myLocation "manifest\Manifest.ps1")
    . (Join-Path $myLocation "converter_util\RegistryOps.ps1")
    . (Join-Path $myLocation "converter_util\VfsOps.ps1")
    . (Join-Path $myLocation "converter_util\CWACNetworkSetup.ps1")

    if ($MakeAppx)
    {
        $makeAppxFullPath = GetMakeAppxFullPath -Logger $logger
    }

    $pruneRulesFile = (Get-Item (Join-Path $myLocation "converter_util\file_pruner_rules.csv")).FullName
    $registryDllPath = (Join-Path $myLocation "registry")
    $commonDllPath = Join-Path $myLocation "common"

    # Clean Input - PowerShell's missing param prompt will leave quotes on strings causing them to not be recognized as paths
    $Installer = $Installer.Replace("`"","")
    $Destination = $Destination.Replace("`"","")
    $PackageName = $PackageName.Replace("`"","")
    $Publisher = $Publisher.Replace("`"","")

    $ExpandedBaseImage = $ExpandedBaseImage.Replace("`"","")

    $logger.NewLogSection("Validating Input", "ValidateInput")

    ValidateInstaller -Installer $Installer -Logger $logger
    ValidateDestination -Destination $Destination -Logger $logger
    if (!([string]::IsNullOrEmpty($ExpandedBaseImage)))
    {
        ValidateExpandedBaseImage -ExpandedBaseImage $ExpandedBaseImage -Logger $logger
    }
    if ($AppExecutable)
    {
        $AppExecutable = $AppExecutable -replace "/", "\"
        ValidateAppExecutable -AppExecutable $AppExecutable -Logger $logger
    }
    if ($NatSubnetPrefix)
    {
        ValidateSubnetPrefix -NatSubnetPrefix $NatSubnetPrefix -Logger $logger
    }

    $installerObj = Get-Item $Installer
    $installerDir = $installerObj.DirectoryName
    $installerName = $installerObj.Name

    $logger.NewLogSection("Setting up Conversion Environment", "SetupEnvironment")

    $networkId = GetOrCreateCWACNetwork -NatSubnetPrefix $NatSubnetPrefix -Logger $logger

    $hostSharedFolderPath = Join-Path $hostTempLocation "shared"
    $logger.LogDiag("Creating folder to share with isolated environment at $hostSharedFolderPath")
    $hostSharedFolder = (New-Item -Path $hostSharedFolderPath -ItemType Directory -Force).FullName
    $logger.LogDiag("Copying contents of $installerDir to $hostSharedFolder to be shared with isolated environment")
    Copy-Item -Path $installerDir\* -Destination $hostSharedFolder -Recurse

    $hostTempDestination = (New-Item -Path (Join-Path $hostTempLocation "output") -ItemType Directory -Force).FullName

    $installerPath = Join-Path $hostSharedFolder $installerName

    $logger.NewLogSection("Initializing AppxManifest.xml", "InitializeAppxManifest")

    $schemaPath = Join-Path $myLocation "manifest\schema"
    $manifest = [AppxManifest]::new($hostTempDestination, $schemaPath)
    $application = $manifest.AddPackageApplication()



    # Required Manifest Properties come from mandatory parameters (non-null)
    #  - If the mandatory parameter fails validation against the manifest schema, an exception will be thrown and the script will exit
    $requiredManifestProperties = New-Object 'System.Collections.Generic.List[System.Collections.Hashtable]'

    $requiredManifestProperties.Add(@{"PropertyName" = "PackageName"; "PropertySetMethod" = $manifest.SetPackageIdentityName;      "Value" = $PackageName; })
    $requiredManifestProperties.Add(@{"PropertyName" = "Publisher";   "PropertySetMethod" = $manifest.SetPackageIdentityPublisher; "Value" = $Publisher; })
    $requiredManifestProperties.Add(@{"PropertyName" = "Version";     "PropertySetMethod" = $manifest.SetPackageIdentityVersion;   "Value" = $Version.ToString(); })

    # if the AppExecutable optional parameter is specified, treat it like a Required Manifest Property (this is a special case)
    if (!([string]::IsNullOrEmpty($AppExecutable)))
    {
        $vfsRelative = ($AppExecutable -split ":")[1]
        $tokenizedVfsRelative = TokenizeVfsRelativePath $registryDllPath $vfsRelative
        $vfsExecutable = Join-Path "VFS" $tokenizedVfsRelative
        $requiredManifestProperties.Add(@{"PropertyName" = "AppExecutable"; "PropertySetMethod" = $application.SetExecutable; "Value" = $vfsExecutable; })
    }

    foreach ($property in $requiredManifestProperties)
    {
        Set-ManifestProperty @property -Logger $logger
    }



    # Optional Manifest Properties come from optional parameters (can be null).
    #  - If the optional parameter is not specified, this manifest property gets a default value taken from Required Manifest Properties instead.
    #  - Else if the specified parameter fails validation agains the manifest schema, an exception will be thrown and the script will exit
    $optionalManifestProperties = New-Object 'System.Collections.Generic.List[System.Collections.Hashtable]'

    $optionalManifestProperties.Add(@{"PropertyName" = "PackagePublisherDisplayName"; "PropertySetMethod" = $manifest.SetPackagePropertiesPublisherDisplayName; "Value" = $PackagePublisherDisplayName; "DefaultValue" = $Publisher; })
    $optionalManifestProperties.Add(@{"PropertyName" = "PackageDisplayName";          "PropertySetMethod" = $manifest.SetPackagePropertiesDisplayName;          "Value" = $PackageDisplayName;          "DefaultValue" = $PackageName; })
    $optionalManifestProperties.Add(@{"PropertyName" = "AppDescription";              "PropertySetMethod" = $application.SetVisualElementsDescription;          "Value" = $AppDescription;              "DefaultValue" = $PackageName; })
    $optionalManifestProperties.Add(@{"PropertyName" = "AppDisplayName";              "PropertySetMethod" = $application.SetVisualElementsDisplayName;          "Value" = $AppDisplayName;              "DefaultValue" = $PackageName; })
    $optionalManifestProperties.Add(@{"PropertyName" = "AppId";                       "PropertySetMethod" = $application.SetId;                                 "Value" = $AppId;                       "DefaultValue" = $PackageName; })

    foreach ($property in $optionalManifestProperties)
    {
        Set-ManifestPropertyWithDefault @property -Logger $logger
    }



    if ($AppFileTypes)
    {
        $logger.LogDiag("Setting application file type assocation in appxmanifest.xml to $AppFileTypes")
        $application.AddFileTypeAssocation($AppFileTypes, "filetypes")
    }

    $logger.NewLogSection("Running Installer in Isolated Environment", "RunInstaller")

    $sequencerArgs = @{
        ExpandedBaseImage = $ExpandedBaseImage
        Installer = $installerPath
        InstallerArguments = $InstallerArguments
        InstallerValidExitCodes = $InstallerValidExitCodes
        ShareFolder = $hostSharedFolder
        Destination = $hostTempDestination
        DllPath = $hcsDllPath
        LogPathToken = "<log_folder>"
        Logger = $logger
    }

    Invoke-Sequencer @sequencerArgs
    $logger.LogDiag("Isolated Installer Run Complete!")

    $logger.NewLogSection("Preparing Virtual Registry", "PrepareRegistry")

    [System.Reflection.Assembly]::LoadFile((Join-Path $commonDllPath "Microsoft.DesktopAppConverter.Common.dll")) > $null
    [System.Reflection.Assembly]::LoadFile((Join-Path $commonDllPath "Microsoft.WindowsAPICodePack.Shell.dll")) > $null
    [System.Reflection.Assembly]::LoadFile((Join-Path $commonDllPath "Microsoft.WindowsAPICodePack.dll")) > $null
    $kfp = [Microsoft.Centennial.Tools.DesktopAppConverter.KnownFoldersPrivate]::LoadFromCurrentEnv()
    $KnownFoldersListFile = Join-Path $hostSharedFolder "KnownFolders.xml"
    $kfp.SaveToXml($KnownFoldersListFile)

    $registryArgs = @{
        DllPath = (Join-Path $myLocation "registry")
        SamplePath = (Join-Path $myLocation "registry\NoiseSample\")
        HivesPath = (Join-Path $hostTempDestination "Hives\")
        Destination = $hostTempDestination
        KnownFoldersListFile = $KnownFoldersListFile
        Logger = $logger
    }
    $logger.LogDiag("Begin Convert-Registry...")
    Convert-Registry @registryArgs


    $logger.NewLogSection("Preparing Virtual File System", "PrepareFileSystem")

    $fileArgs = @{
        MetadataProcessorExecutable = (Join-Path $myLocation "metadata_processor\MetadataProcessor.exe")
        Destination = $hostTempDestination
        PruneRulesFile = $pruneRulesFile
        KnownFoldersListFile = $KnownFoldersListFile
        Logger = $logger
    }
    $logger.LogDiag("Begin Convert-Files...")
    $vfs = Convert-Files @fileArgs | Select-Object -Last 1

    $logger.NewLogSection("Finalizing AppxManifest.xml", "FinalizeAppxManifest")

    # Initialize to dummy executable
    $manifestExecutable = "SampleApp.exe"
    $isAppExecutableFound = $False

    # if no executable was given, try to find it using the shortcut made in the VFS
    if ([string]::IsNullOrEmpty($AppExecutable))
    {
        $logger.LogDiag("AppExecutable param was not specified, but is required in the AppxManifest.xml in order for your app to run. Attempting to discover it by searching the VFS for a Start Menu shortcut.")
        $AppExecutable = Find-Executable -VFS $vfs -Logger $logger

    if (!([string]::IsNullOrEmpty($AppExecutable))) {
        $vfsExecutable = ($AppExecutable -split ':')[1]

            $logger.LogDiag("A shortcut was found in the VFS that targets $vfsExecutable. Testing that path to ensure the executable is there.")

            $tokenizedVfsRelative = TokenizeVfsRelativePath $registryDllPath $vfsExecutable
            $fullVFSExecutable = Join-Path $vfs $tokenizedVfsRelative
            if (!(Test-Path $fullVFSExecutable))
        {
            $logger.LogWarning("AppExecutable: warning 4200: Application executable at $vfsExecutable could not be found. Please fix the application executable property in the appxmanifest")
        }
    else
    {
                $isAppExecutableFound = $True
                $manifestExecutable = Join-Path "VFS" $tokenizedVfsRelative

                $logger.LogDiag("The executable path targeted by the discovered shortcut ($vfsExecutable) is a valid path. Attempting to set application executable to $manifestExecutable in the appx manifest.")

                try
    {
                    $application.SetExecutable($manifestExecutable)
    }
                catch
    {
                    $logger.LogWarning("AppExecutable: warning 4230: Appx Manifest validation failed when attempting to set Application Executable to $manifestExecutable. Please manually edit the manifest with a path to your executable.")
    }
    }
    }
    else
    {
            $logger.LogWarning("AppExecutable: warning 4210: Executable could not be determined from shortcut. Please manually edit the manifest with a path to your executable.")
    }
    }

    $logger.LogDiag("Saving manifest...")
    $manifest.Save()

    $logger.NewLogSection("Preparing Final Output", "FinalOutput")

    $logger.LogDiag("Copying placeholder Assets into package...")
    $assetsPath = (Join-Path $myLocation "manifest\assets")
    Copy-Item $assetsPath (Join-Path $hostTempDestination "Assets") -Recurse
    $logger.LogDiag("Please replace placeholder assets with your own by changing the files contained at $(Join-Path $Destination "Assets")")

    $logger.LogDiag("Copying remaining output into the final output destination...")
    Get-ChildItem -Path $hostTempDestination -Recurse | Move-Item -Destination $Destination

    if ($MakeAppx)
    {
        $logger.NewLogSection("Making Final Appx Package", "FinalAppx")

        if ($application.IsExecutableChangedFromDefault())
        {
            $destinationFile = (Join-Path $Destination "$PackageName.appx")
            $makeAppxLog = Join-Path $logDirectory "MakeAppx.log"
            & $makeAppxFullPath pack /d $Destination /p $destinationFile | Out-File -FilePath $makeAppxLog -Append -Force
            if (!$?)
            {
                throw "MakeAppx: fatal error 1300: error calling MakeAppx on your package. See $makeAppxLog for details."
            }
            else
            {
                $logger.LogDiag("Appx creation complete. Find it at $destinationFile")
                $logger.SendFileSize("Appx", $destinationFile)
            }
        }
        else
        {
            throw "MakeAppx: fatal error 1310: Unable to create appx package. AppExecutable property in AppxManifest.xml needs to be replaced before calling MakeAppx."
        }
    }

    $logger.LogDiag("Conversion complete. Your converted application is located at $Destination.")

    $logger.SendCompleted();
}
catch
{
    $lastError = $_
    $logger.LogDiag("An error occurred. Refer to logs in $logDirectory")
    $logger.LogDiag($lastError) 4> $null
    $logger.LogDiag($lastError.ScriptStackTrace) 4> $null
    $logger.SendFatalException($lastError)
    throw $lastError
}

# SIG # Begin signature block
# MIIkJAYJKoZIhvcNAQcCoIIkFTCCJBECAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDS3TDZNfAMoIwW
# vlYCezrYHp5l5Ho32+H+LlLXQmwHvaCCDZIwggYQMIID+KADAgECAhMzAAAAZEeE
# lIbbQRk4AAAAAABkMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMTUxMDI4MjAzMTQ2WhcNMTcwMTI4MjAzMTQ2WjCBgzEL
# MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
# bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9Q
# UjEeMBwGA1UEAxMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMIIBIjANBgkqhkiG9w0B
# AQEFAAOCAQ8AMIIBCgKCAQEAky7a2OY+mNkbD2RfTahYTRQ793qE/DwRMTrvicJK
# LUGlSF3dEp7vq2YoNNV9KlV7TE2K8sDxstNSFYu2swi4i1AL3X/7agmg3GcExPHf
# vHUYIEC+eCyZVt3u9S7dPkL5Wh8wrgEUirCCtVGg4m1l/vcYCo0wbU06p8XzNi3u
# XyygkgCxHEziy/f/JCV/14/A3ZduzrIXtsccRKckyn6B5uYxuRbZXT7RaO6+zUjQ
# hiyu3A4hwcCKw+4bk1kT9sY7gHIYiFP7q78wPqB3vVKIv3rY6LCTraEbjNR+phBQ
# EL7hyBxk+ocu+8RHZhbAhHs2r1+6hURsAg8t4LAOG6I+JQIDAQABo4IBfzCCAXsw
# HwYDVR0lBBgwFgYIKwYBBQUHAwMGCisGAQQBgjdMCAEwHQYDVR0OBBYEFFhWcQTw
# vbsz9YNozOeARvdXr9IiMFEGA1UdEQRKMEikRjBEMQ0wCwYDVQQLEwRNT1BSMTMw
# MQYDVQQFEyozMTY0Mis0OWU4YzNmMy0yMzU5LTQ3ZjYtYTNiZS02YzhjNDc1MWM0
# YjYwHwYDVR0jBBgwFoAUSG5k5VAF04KqFzc3IrVtqMp1ApUwVAYDVR0fBE0wSzBJ
# oEegRYZDaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljQ29k
# U2lnUENBMjAxMV8yMDExLTA3LTA4LmNybDBhBggrBgEFBQcBAQRVMFMwUQYIKwYB
# BQUHMAKGRWh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY2VydHMvTWlj
# Q29kU2lnUENBMjAxMV8yMDExLTA3LTA4LmNydDAMBgNVHRMBAf8EAjAAMA0GCSqG
# SIb3DQEBCwUAA4ICAQCI4gxkQx3dXK6MO4UktZ1A1r1mrFtXNdn06DrARZkQTdu0
# kOTLdlGBCfCzk0309RLkvUgnFKpvLddrg9TGp3n80yUbRsp2AogyrlBU+gP5ggHF
# i7NjGEpj5bH+FDsMw9PygLg8JelgsvBVudw1SgUt625nY7w1vrwk+cDd58TvAyJQ
# FAW1zJ+0ySgB9lu2vwg0NKetOyL7dxe3KoRLaztUcqXoYW5CkI+Mv3m8HOeqlhyf
# FTYxPB5YXyQJPKQJYh8zC9b90JXLT7raM7mQ94ygDuFmlaiZ+QSUR3XVupdEngrm
# ZgUB5jX13M+Pl2Vv7PPFU3xlo3Uhj1wtupNC81epoxGhJ0tRuLdEajD/dCZ0xIni
# esRXCKSC4HCL3BMnSwVXtIoj/QFymFYwD5+sAZuvRSgkKyD1rDA7MPcEI2i/Bh5O
# MAo9App4sR0Gp049oSkXNhvRi/au7QG6NJBTSBbNBGJG8Qp+5QThKoQUk8mj0ugr
# 4yWRsA9JTbmqVw7u9suB5OKYBMUN4hL/yI+aFVsE/KJInvnxSzXJ1YHka45ADYMK
# AMl+fLdIqm3nx6rIN0RkoDAbvTAAXGehUCsIod049A1T3IJyUJXt3OsTd3WabhIB
# XICYfxMg10naaWcyUePgW3+VwP0XLKu4O1+8ZeGyaDSi33GnzmmyYacX3BTqMDCC
# B3owggVioAMCAQICCmEOkNIAAAAAAAMwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
# HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29m
# dCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDExMB4XDTExMDcwODIwNTkw
# OVoXDTI2MDcwODIxMDkwOVowfjELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
# bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
# b3JhdGlvbjEoMCYGA1UEAxMfTWljcm9zb2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAx
# MTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAKvw+nIQHC6t2G6qghBN
# NLrytlghn0IbKmvpWlCquAY4GgRJun/DDB7dN2vGEtgL8DjCmQawyDnVARQxQtOJ
# DXlkh36UYCRsr55JnOloXtLfm1OyCizDr9mpK656Ca/XllnKYBoF6WZ26DJSJhIv
# 56sIUM+zRLdd2MQuA3WraPPLbfM6XKEW9Ea64DhkrG5kNXimoGMPLdNAk/jj3gcN
# 1Vx5pUkp5w2+oBN3vpQ97/vjK1oQH01WKKJ6cuASOrdJXtjt7UORg9l7snuGG9k+
# sYxd6IlPhBryoS9Z5JA7La4zWMW3Pv4y07MDPbGyr5I4ftKdgCz1TlaRITUlwzlu
# ZH9TupwPrRkjhMv0ugOGjfdf8NBSv4yUh7zAIXQlXxgotswnKDglmDlKNs98sZKu
# HCOnqWbsYR9q4ShJnV+I4iVd0yFLPlLEtVc/JAPw0XpbL9Uj43BdD1FGd7P4AOG8
# rAKCX9vAFbO9G9RVS+c5oQ/pI0m8GLhEfEXkwcNyeuBy5yTfv0aZxe/CHFfbg43s
# TUkwp6uO3+xbn6/83bBm4sGXgXvt1u1L50kppxMopqd9Z4DmimJ4X7IvhNdXnFy/
# dygo8e1twyiPLI9AN0/B4YVEicQJTMXUpUMvdJX3bvh4IFgsE11glZo+TzOE2rCI
# F96eTvSWsLxGoGyY0uDWiIwLAgMBAAGjggHtMIIB6TAQBgkrBgEEAYI3FQEEAwIB
# ADAdBgNVHQ4EFgQUSG5k5VAF04KqFzc3IrVtqMp1ApUwGQYJKwYBBAGCNxQCBAwe
# CgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0j
# BBgwFoAUci06AjGQQ7kUBU7h6qfHMdEjiTQwWgYDVR0fBFMwUTBPoE2gS4ZJaHR0
# cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9vQ2Vy
# QXV0MjAxMV8yMDExXzAzXzIyLmNybDBeBggrBgEFBQcBAQRSMFAwTgYIKwYBBQUH
# MAKGQmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2Vy
# QXV0MjAxMV8yMDExXzAzXzIyLmNydDCBnwYDVR0gBIGXMIGUMIGRBgkrBgEEAYI3
# LgMwgYMwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lv
# cHMvZG9jcy9wcmltYXJ5Y3BzLmh0bTBABggrBgEFBQcCAjA0HjIgHQBMAGUAZwBh
# AGwAXwBwAG8AbABpAGMAeQBfAHMAdABhAHQAZQBtAGUAbgB0AC4gHTANBgkqhkiG
# 9w0BAQsFAAOCAgEAZ/KGpZjgVHkaLtPYdGcimwuWEeFjkplCln3SeQyQwWVfLiw+
# +MNy0W2D/r4/6ArKO79HqaPzadtjvyI1pZddZYSQfYtGUFXYDJJ80hpLHPM8QotS
# 0LD9a+M+By4pm+Y9G6XUtR13lDni6WTJRD14eiPzE32mkHSDjfTLJgJGKsKKELuk
# qQUMm+1o+mgulaAqPyprWEljHwlpblqYluSD9MCP80Yr3vw70L01724lruWvJ+3Q
# 3fMOr5kol5hNDj0L8giJ1h/DMhji8MUtzluetEk5CsYKwsatruWy2dsViFFFWDgy
# cScaf7H0J/jeLDogaZiyWYlobm+nt3TDQAUGpgEqKD6CPxNNZgvAs0314Y9/HG8V
# fUWnduVAKmWjw11SYobDHWM2l4bf2vP48hahmifhzaWX0O5dY0HjWwechz4GdwbR
# BrF1HxS+YWG18NzGGwS+30HHDiju3mUv7Jf2oVyW2ADWoUa9WfOXpQlLSBCZgB/Q
# ACnFsZulP0V3HjXG0qKin3p6IvpIlR+r+0cjgPWe+L9rt0uX4ut1eBrs6jeZeRhL
# /9azI2h15q/6/IvrC4DqaTuv/DDtBEyO3991bWORPdGdVk5Pv4BXIqF4ETIheu9B
# CrE/+6jMpF3BoYibV3FWTkhFwELJm3ZbCoBIa/15n8G9bW1qyVJzEw16UM0xghXo
# MIIV5AIBATCBlTB+MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
# MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
# MSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQSAyMDExAhMzAAAA
# ZEeElIbbQRk4AAAAAABkMA0GCWCGSAFlAwQCAQUAoIHWMBkGCSqGSIb3DQEJAzEM
# BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMC8GCSqG
# SIb3DQEJBDEiBCDK8dAmCYFP3KmTe3b3m44+lEVP0J2Y+OIcgou5NksX7TBqBgor
# BgEEAYI3AgEMMVwwWqAugCwAVwBpAG4AZABvAHcAcwAgAFAAaABvAG4AZQAgAFAA
# YQBjAGsAYQBnAGUAc6EogCZodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vd2luZG93
# c3Bob25lLzANBgkqhkiG9w0BAQEFAASCAQBJLSsik8Yl4QnNDIuWODFNiuxB5+GF
# 1G9oc+2+Yvf4GQJIhke7zNjE0CKRoRMeorQxakkC4z+2ZuXP80qsqBQkPq2WrrXv
# YKxXb023nEuCc6qb2SGcZhpMPsYKuXQOVT0ifN73rSDXdHa3FJZ7/w5ZVGSZjtAE
# B8/AyIYE5cEEtNLG6S/rCPpHqV210ul2Hw1ijBiWyvUql6u72V91sc3eMS9ZJqvv
# oO5ydRAajC/tldynzRCPJbupPprySmLW1YLNLVVVKdOCZn8xoNW3qb2vLOMQS6Zq
# N8otEwUbjyf5cuReoNST+a0oOo98GblGXD+CjCMm8dUqiIR1t0TfcLDYoYITSjCC
# E0YGCisGAQQBgjcDAwExghM2MIITMgYJKoZIhvcNAQcCoIITIzCCEx8CAQMxDzAN
# BglghkgBZQMEAgEFADCCAT0GCyqGSIb3DQEJEAEEoIIBLASCASgwggEkAgEBBgor
# BgEEAYRZCgMBMDEwDQYJYIZIAWUDBAIBBQAEIHGDHV2/SFF7wxljYmNnlpfbDokc
# kYbvct/26lW2rp7qAgZW9iAkWLcYEzIwMTYwNDA2MDQxNDEwLjMzOVowBwIBAYAC
# AfSggbmkgbYwgbMxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
# DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
# DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIgRFNFIEVTTjo3RDJFLTM3
# ODItQjBGNzElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaCC
# Ds0wggZxMIIEWaADAgECAgphCYEqAAAAAAACMA0GCSqGSIb3DQEBCwUAMIGIMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNy
# b3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0xMDA3MDEy
# MTM2NTVaFw0yNTA3MDEyMTQ2NTVaMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAy
# MDEwMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAqR0NvHcRijog7PwT
# l/X6f2mUa3RUENWlCgCChfvtfGhLLF/Fw+Vhwna3PmYrW/AVUycEMR9BGxqVHc4J
# E458YTBZsTBED/FgiIRUQwzXTbg4CLNC3ZOs1nMwVyaCo0UN0Or1R4HNvyRgMlhg
# RvJYR4YyhB50YWeRX4FUsc+TTJLBxKZd0WETbijGGvmGgLvfYfxGwScdJGcSchoh
# iq9LZIlQYrFd/XcfPfBXday9ikJNQFHRD5wGPmd/9WbAA5ZEfu/QS/1u5ZrKsajy
# eioKMfDaTgaRtogINeh4HLDpmc085y9Euqf03GS9pAHBIAmTeM38vMDJRF1eFpwB
# BU8iTQIDAQABo4IB5jCCAeIwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFNVj
# OlyKMZDzQ3t8RhvFM2hahW1VMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsG
# A1UdDwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFNX2VsuP6KJc
# YmjRPZSQW9fOmhjEMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWljcm9z
# b2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIz
# LmNybDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWlj
# cm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3J0
# MIGgBgNVHSABAf8EgZUwgZIwgY8GCSsGAQQBgjcuAzCBgTA9BggrBgEFBQcCARYx
# aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL1BLSS9kb2NzL0NQUy9kZWZhdWx0Lmh0
# bTBABggrBgEFBQcCAjA0HjIgHQBMAGUAZwBhAGwAXwBQAG8AbABpAGMAeQBfAFMA
# dABhAHQAZQBtAGUAbgB0AC4gHTANBgkqhkiG9w0BAQsFAAOCAgEAB+aIUQ3ixuCY
# P4FxAz2do6Ehb7Prpsz1Mb7PBeKp/vpXbRkws8LFZslq3/Xn8Hi9x6ieJeP5vO1r
# VFcIK1GCRBL7uVOMzPRgEop2zEBAQZvcXBf/XPleFzWYJFZLdO9CEMivv3/Gf/I3
# fVo/HPKZeUqRUgCvOA8X9S95gWXZqbVr5MfO9sp6AG9LMEQkIjzP7QOllo9ZKby2
# /QThcJ8ySif9Va8v/rbljjO7Yl+a21dA6fHOmWaQjP9qYn/dxUoLkSbiOewZSnFj
# nXshbcOco6I8+n99lmqQeKZt0uGc+R38ONiU9MalCpaGpL2eGq4EQoO4tYCbIjgg
# tSXlZOz39L9+Y1klD3ouOVd2onGqBooPiRa6YacRy5rYDkeagMXQzafQ732D8OE7
# cQnfXXSYIghh2rBQHm+98eEA3+cxB6STOvdlR3jo+KhIq/fecn5ha293qYHLpwms
# ObvsxsvYgrRyzR30uIUBHoD7G4kqVDmyW9rIDVWZeodzOwjmmC3qjeAzLhIp9cAv
# VCch98isTtoouLGp25ayp0Kiyc8ZQU3ghvkqmqMRZjDTu3QyS99je/WZii8bxyGv
# WbWu3EQ8l1Bx16HSxVXjad5XwdHeMMD9zOZN+w2/XU/pnR4ZOC+8z1gFLu8NoFA1
# 2u8JJxzVs341Hgi62jbb01+P3nSISRIwggTaMIIDwqADAgECAhMzAAAAdrYQ4XyH
# IzwiAAAAAAB2MA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
# EwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
# ZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBD
# QSAyMDEwMB4XDTE1MTAwNzE4MTc0MFoXDTE3MDEwNzE4MTc0MFowgbMxCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
# HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xDTALBgNVBAsTBE1PUFIxJzAl
# BgNVBAsTHm5DaXBoZXIgRFNFIEVTTjo3RDJFLTM3ODItQjBGNzElMCMGA1UEAxMc
# TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIwDQYJKoZIhvcNAQEBBQAD
# ggEPADCCAQoCggEBAKMFk/N0yGDhLMtQW3kOqJ4Z0I/SJKMGYbpWlrU6lAGi4Gfd
# 6REj/v6CYhk/DbyHJ5LmZlFQrve1EfsGDqij9yZbg+/CbljqTchT65F4EHXGBr6D
# kMBpuuhZd3roWnayqmIDcc9TeNeWJ9iXfreEgL/MOuhW3jQNsUX8CIA7kvwKNcr8
# U/UIcR8qZvRgVo4dMhxkH1yhaSU5lR0qVndwh75w4MbFbRg5pkP5TuoSLQauZ24x
# 6oysX7IPxnsWCdNj8GHrDlVj9U2qAMgZ07S6UQItrZKBbmyFZkaS86MtU1RKCUHS
# 48sojm2+WGQp63codmL/YXzO9MPTwb3lncFjIYMCAwEAAaOCARswggEXMB0GA1Ud
# DgQWBBT0FIVIPPf676xoDtCRqcertABcyzAfBgNVHSMEGDAWgBTVYzpcijGQ80N7
# fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29m
# dC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNUaW1TdGFQQ0FfMjAxMC0wNy0wMS5j
# cmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jv
# c29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0YVBDQV8yMDEwLTA3LTAxLmNydDAM
# BgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0GCSqGSIb3DQEBCwUA
# A4IBAQBatESgWH1l3/tDq3qZmgvO3XSwYE5Pm6ueYySASN9K26lyAhyD2h0ZnSiW
# CkvrcaqqL0CV5lKHih4Pj3MpNZNjxYx0OHLatEfBiHj92uIs1kmbkfKbe1lsdmSH
# ufhbX23D1+Tlh5LWJKLhiwehYhVOvb6/+wVx3kAX+hkJPVd12xMnZXczCME/sNT9
# ibrlnJFSs6G4vNx/PTa7bXFJhKqFPNsGgGK/mPxPtwXdTgVJO/orfSzVmkHgC3iX
# lXFKdCjxOQvIRqwAOELfyS15I38n6FelzU3Y7obGqOtXtcyAgOh0fecCJ1PRnCFz
# RMUCRxBroucAUxSk/he5M9n/eGzIoYIDdjCCAl4CAQEwgeOhgbmkgbYwgbMxCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xDTALBgNVBAsTBE1PUFIx
# JzAlBgNVBAsTHm5DaXBoZXIgRFNFIEVTTjo3RDJFLTM3ODItQjBGNzElMCMGA1UE
# AxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaIlCgEBMAkGBSsOAwIaBQAD
# FQDeHTu9FPbcUDPm0TdIfTmT4JN3L6CBwjCBv6SBvDCBuTELMAkGA1UEBhMCVVMx
# EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoT
# FU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9QUjEnMCUGA1UECxMe
# bkNpcGhlciBOVFMgRVNOOjU3RjYtQzFFMC01NTRDMSswKQYDVQQDEyJNaWNyb3Nv
# ZnQgVGltZSBTb3VyY2UgTWFzdGVyIENsb2NrMA0GCSqGSIb3DQEBBQUAAgUA2q7W
# nTAiGA8yMDE2MDQwNjAwMjgxM1oYDzIwMTYwNDA3MDAyODEzWjB0MDoGCisGAQQB
# hFkKBAExLDAqMAoCBQDartadAgEAMAcCAQACAhuGMAcCAQACAh4xMAoCBQDasCgd
# AgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkKAwGgCjAIAgEAAgMW42Ch
# CjAIAgEAAgMHoSAwDQYJKoZIhvcNAQEFBQADggEBAJc39/Sw9dEjJCpjVcGslxRz
# qxNEI4H/nIxxfcO79uoc9l6Cd2YYg7SlHACRuOuSaUOAeOrDDr7G/I+XMuOyFFFT
# ozqZwikisRrekdx/AvUjYeuL7UKR4Sscf4uNcZoh6c5QTwxLaYctH7mb8k+9xWlQ
# ycgB7bRwLRTXQN/q86jFCU3iPMC6lxhbf2T3BO8yqgcOSEdVqqyfK1bvS6MXmrQp
# KZk6ycaLfoR+raTNSJajS7unG36eHDhhyy8U2I1UjG6PSZrA25G2u+ZODH30iHqg
# 5DAmgW+kDg+N43NDFvETN1Hwo+taQZIiCa4BCklAEo8nCpyoTJhM2gDG3ymtCq4x
# ggL1MIIC8QIBATCBkzB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAA
# AHa2EOF8hyM8IgAAAAAAdjANBglghkgBZQMEAgEFAKCCATIwGgYJKoZIhvcNAQkD
# MQ0GCyqGSIb3DQEJEAEEMC8GCSqGSIb3DQEJBDEiBCAo4wF5R/DmpWJwBGqJPNxj
# R187eeApJKcWi6wmpZ9kiDCB4gYLKoZIhvcNAQkQAgwxgdIwgc8wgcwwgbEEFN4d
# O70U9txQM+bRN0h9OZPgk3cvMIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNV
# BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
# c29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAg
# UENBIDIwMTACEzMAAAB2thDhfIcjPCIAAAAAAHYwFgQUyq045YBDBB1qOB/jJr2t
# QD50qmIwDQYJKoZIhvcNAQELBQAEggEASRaIqBzQaFS5E+qqNW7+5+1eikGdj0Ul
# VOL37wnNiLUkICcfDlgPTaeRBqmlhBQPb5dgg+6MvSIMf0f76O6UCz8nqL9m8I6G
# ni6JZAv/z8/cgnyOan138LpdZsv0gWIWmV/bSLvt/sfolxmEduAocSG7FQY78zQv
# MFcnFJrTNmzIFB1O0m//pEiKOoINWbbK2h3KtRoD52bMyvn21Ott2SS2znnJOUZs
# VJyv8cV5rDlvkYCswt+WXiWgym0XTOGXa2QI6pkqLyePLO12o9rn3snyRgo1i2/R
# mxwLyu4d68QWCqeasnjU425daUwGGkb1eWYdCANzB6W9RXJPro+vsg==
# SIG # End signature block
