/*
 * Decompiled with CFR 0.152.
 */
package nl.siegmann.epublib.epub;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import javax.xml.parsers.ParserConfigurationException;
import nl.siegmann.epublib.domain.Book;
import nl.siegmann.epublib.domain.Guide;
import nl.siegmann.epublib.domain.GuideReference;
import nl.siegmann.epublib.domain.MediaType;
import nl.siegmann.epublib.domain.Resource;
import nl.siegmann.epublib.domain.Resources;
import nl.siegmann.epublib.domain.Spine;
import nl.siegmann.epublib.domain.SpineReference;
import nl.siegmann.epublib.epub.DOMUtil;
import nl.siegmann.epublib.epub.EpubReader;
import nl.siegmann.epublib.epub.PackageDocumentBase;
import nl.siegmann.epublib.epub.PackageDocumentMetadataReader;
import nl.siegmann.epublib.service.MediatypeService;
import nl.siegmann.epublib.util.ResourceUtil;
import nl.siegmann.epublib.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class PackageDocumentReader
extends PackageDocumentBase {
    private static final Logger log = LoggerFactory.getLogger(PackageDocumentReader.class);
    private static final String[] POSSIBLE_NCX_ITEM_IDS = new String[]{"toc", "ncx"};

    public static void read(Resource packageResource, EpubReader epubReader, Book book, Resources resources) throws UnsupportedEncodingException, SAXException, IOException, ParserConfigurationException {
        Document packageDocument = ResourceUtil.getAsDocument(packageResource);
        String packageHref = packageResource.getHref();
        resources = PackageDocumentReader.fixHrefs(packageHref, resources);
        PackageDocumentReader.readGuide(packageDocument, epubReader, book, resources);
        HashMap<String, String> idMapping = new HashMap<String, String>();
        resources = PackageDocumentReader.readManifest(packageDocument, packageHref, epubReader, resources, idMapping);
        book.setResources(resources);
        PackageDocumentReader.readCover(packageDocument, book);
        book.setMetadata(PackageDocumentMetadataReader.readMetadata(packageDocument));
        book.setSpine(PackageDocumentReader.readSpine(packageDocument, epubReader, book.getResources(), idMapping));
        if (book.getCoverPage() == null && book.getSpine().size() > 0) {
            book.setCoverPage(book.getSpine().getResource(0));
        }
    }

    private static Resources readManifest(Document packageDocument, String packageHref, EpubReader epubReader, Resources resources, Map<String, String> idMapping) {
        Element manifestElement = DOMUtil.getFirstElementByTagNameNS(packageDocument.getDocumentElement(), "http://www.idpf.org/2007/opf", "manifest");
        Resources result = new Resources();
        if (manifestElement == null) {
            log.error("Package document does not contain element manifest");
            return result;
        }
        NodeList itemElements = manifestElement.getElementsByTagNameNS("http://www.idpf.org/2007/opf", "item");
        int i = 0;
        while (i < itemElements.getLength()) {
            Element itemElement = (Element)itemElements.item(i);
            String id = DOMUtil.getAttribute(itemElement, "http://www.idpf.org/2007/opf", "id");
            String href = DOMUtil.getAttribute(itemElement, "http://www.idpf.org/2007/opf", "href");
            try {
                href = URLDecoder.decode(href, "UTF-8");
            }
            catch (UnsupportedEncodingException e) {
                log.error(e.getMessage());
            }
            String mediaTypeName = DOMUtil.getAttribute(itemElement, "http://www.idpf.org/2007/opf", "media-type");
            Resource resource = resources.remove(href);
            if (resource == null) {
                log.error("resource with href '" + href + "' not found");
            } else {
                resource.setId(id);
                MediaType mediaType = MediatypeService.getMediaTypeByName(mediaTypeName);
                if (mediaType != null) {
                    resource.setMediaType(mediaType);
                }
                result.add(resource);
                idMapping.put(id, resource.getId());
            }
            ++i;
        }
        return result;
    }

    private static void readGuide(Document packageDocument, EpubReader epubReader, Book book, Resources resources) {
        Element guideElement = DOMUtil.getFirstElementByTagNameNS(packageDocument.getDocumentElement(), "http://www.idpf.org/2007/opf", "guide");
        if (guideElement == null) {
            return;
        }
        Guide guide = book.getGuide();
        NodeList guideReferences = guideElement.getElementsByTagNameNS("http://www.idpf.org/2007/opf", "reference");
        int i = 0;
        while (i < guideReferences.getLength()) {
            Element referenceElement = (Element)guideReferences.item(i);
            String resourceHref = DOMUtil.getAttribute(referenceElement, "http://www.idpf.org/2007/opf", "href");
            if (!StringUtil.isBlank(resourceHref)) {
                Resource resource = resources.getByHref(StringUtil.substringBefore(resourceHref, '#'));
                if (resource == null) {
                    log.error("Guide is referencing resource with href " + resourceHref + " which could not be found");
                } else {
                    String type = DOMUtil.getAttribute(referenceElement, "http://www.idpf.org/2007/opf", "type");
                    if (StringUtil.isBlank(type)) {
                        log.error("Guide is referencing resource with href " + resourceHref + " which is missing the 'type' attribute");
                    } else {
                        String title = DOMUtil.getAttribute(referenceElement, "http://www.idpf.org/2007/opf", "title");
                        if (!GuideReference.COVER.equalsIgnoreCase(type)) {
                            GuideReference reference = new GuideReference(resource, type, title, StringUtil.substringAfter(resourceHref, '#'));
                            guide.addReference(reference);
                        }
                    }
                }
            }
            ++i;
        }
    }

    private static Resources fixHrefs(String packageHref, Resources resourcesByHref) {
        int lastSlashPos = packageHref.lastIndexOf(47);
        if (lastSlashPos < 0) {
            return resourcesByHref;
        }
        Resources result = new Resources();
        for (Resource resource : resourcesByHref.getAll()) {
            if (StringUtil.isNotBlank(resource.getHref()) || resource.getHref().length() > lastSlashPos) {
                resource.setHref(resource.getHref().substring(lastSlashPos + 1));
            }
            result.add(resource);
        }
        return result;
    }

    private static Spine readSpine(Document packageDocument, EpubReader epubReader, Resources resources, Map<String, String> idMapping) {
        Element spineElement = DOMUtil.getFirstElementByTagNameNS(packageDocument.getDocumentElement(), "http://www.idpf.org/2007/opf", "spine");
        if (spineElement == null) {
            log.error("Element spine not found in package document, generating one automatically");
            return PackageDocumentReader.generateSpineFromResources(resources);
        }
        Spine result = new Spine();
        result.setTocResource(PackageDocumentReader.findTableOfContentsResource(spineElement, resources));
        NodeList spineNodes = packageDocument.getElementsByTagNameNS("http://www.idpf.org/2007/opf", "itemref");
        ArrayList<SpineReference> spineReferences = new ArrayList<SpineReference>(spineNodes.getLength());
        int i = 0;
        while (i < spineNodes.getLength()) {
            Element spineItem = (Element)spineNodes.item(i);
            String itemref = DOMUtil.getAttribute(spineItem, "http://www.idpf.org/2007/opf", "idref");
            if (StringUtil.isBlank(itemref)) {
                log.error("itemref with missing or empty idref");
            } else {
                Resource resource;
                String id = idMapping.get(itemref);
                if (id == null) {
                    id = itemref;
                }
                if ((resource = resources.getByIdOrHref(id)) == null) {
                    log.error("resource with id '" + id + "' not found");
                } else {
                    SpineReference spineReference = new SpineReference(resource);
                    if ("no".equalsIgnoreCase(DOMUtil.getAttribute(spineItem, "http://www.idpf.org/2007/opf", "linear"))) {
                        spineReference.setLinear(false);
                    }
                    spineReferences.add(spineReference);
                }
            }
            ++i;
        }
        result.setSpineReferences(spineReferences);
        return result;
    }

    private static Spine generateSpineFromResources(Resources resources) {
        Spine result = new Spine();
        ArrayList<String> resourceHrefs = new ArrayList<String>();
        resourceHrefs.addAll(resources.getAllHrefs());
        Collections.sort(resourceHrefs, String.CASE_INSENSITIVE_ORDER);
        for (String resourceHref : resourceHrefs) {
            Resource resource = resources.getByHref(resourceHref);
            if (resource.getMediaType() == MediatypeService.NCX) {
                result.setTocResource(resource);
                continue;
            }
            if (resource.getMediaType() != MediatypeService.XHTML) continue;
            result.addSpineReference(new SpineReference(resource));
        }
        return result;
    }

    private static Resource findTableOfContentsResource(Element spineElement, Resources resources) {
        String tocResourceId = DOMUtil.getAttribute(spineElement, "http://www.idpf.org/2007/opf", "toc");
        Resource tocResource = null;
        if (StringUtil.isNotBlank(tocResourceId)) {
            tocResource = resources.getByIdOrHref(tocResourceId);
        }
        if (tocResource != null) {
            return tocResource;
        }
        int i = 0;
        while (i < POSSIBLE_NCX_ITEM_IDS.length) {
            tocResource = resources.getByIdOrHref(POSSIBLE_NCX_ITEM_IDS[i]);
            if (tocResource != null) {
                return tocResource;
            }
            tocResource = resources.getByIdOrHref(POSSIBLE_NCX_ITEM_IDS[i].toUpperCase());
            if (tocResource != null) {
                return tocResource;
            }
            ++i;
        }
        tocResource = resources.findFirstResourceByMediaType(MediatypeService.NCX);
        if (tocResource == null) {
            log.error("Could not find table of contents resource. Tried resource with id '" + tocResourceId + "', " + "toc" + ", " + "toc".toUpperCase() + " and any NCX resource.");
        }
        return tocResource;
    }

    static Set<String> findCoverHrefs(Document packageDocument) {
        String coverHref;
        HashSet<String> result = new HashSet<String>();
        String coverResourceId = DOMUtil.getFindAttributeValue(packageDocument, "http://www.idpf.org/2007/opf", "meta", "name", "cover", "content");
        if (StringUtil.isNotBlank(coverResourceId)) {
            coverHref = DOMUtil.getFindAttributeValue(packageDocument, "http://www.idpf.org/2007/opf", "item", "id", coverResourceId, "href");
            if (StringUtil.isNotBlank(coverHref)) {
                result.add(coverHref);
            } else {
                result.add(coverResourceId);
            }
        }
        if (StringUtil.isNotBlank(coverHref = DOMUtil.getFindAttributeValue(packageDocument, "http://www.idpf.org/2007/opf", "reference", "type", "cover", "href"))) {
            result.add(coverHref);
        }
        return result;
    }

    private static void readCover(Document packageDocument, Book book) {
        Set<String> coverHrefs = PackageDocumentReader.findCoverHrefs(packageDocument);
        for (String coverHref : coverHrefs) {
            Resource resource = book.getResources().getByHref(coverHref);
            if (resource == null) {
                log.error("Cover resource " + coverHref + " not found");
                continue;
            }
            if (resource.getMediaType() == MediatypeService.XHTML) {
                book.setCoverPage(resource);
                continue;
            }
            if (!MediatypeService.isBitmapImage(resource.getMediaType())) continue;
            book.setCoverImage(resource);
        }
    }
}

